/**
* \file: partition_metadata.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <string.h>

#include "model/partition_metadata.h"

typedef struct partition_metadata_t
{
	char *identifier;
	char *mount_src;
	char *mount_fs;
	int partition_no;
	partition_metadata_free_func_t free_func;
} partition_metadata_t;

static void partition_metadata_default_free(partition_metadata_t *metadata);

static void partition_metadata_free_elements(partition_metadata_t *metadata);

//-------------------------------------------- API members -------------------------------------------------------
partition_metadata_t *partition_metadata_new(const char *identifier, const char *mountsrc,
		const char *mountfs, int partition_no)
{
	return partition_metadata_flexible_new(identifier,mountsrc,mountfs,partition_no,
			sizeof(partition_metadata_t), partition_metadata_default_free);
}

partition_metadata_t *partition_metadata_flexible_new(const char *identifier, const char *mountsrc,
		const char *mountfs,int partition_no,size_t metadata_size, partition_metadata_free_func_t free_func)
{
	partition_metadata_t *metadata;
	metadata=malloc(metadata_size);
	if (metadata!=NULL)
	{
		metadata->identifier=strdup(identifier);
		metadata->mount_fs=strdup(mountfs);
		metadata->mount_src=strdup(mountsrc);
		metadata->partition_no=partition_no;
		metadata->free_func=free_func;

		if (metadata->identifier==NULL ||
			metadata->mount_fs==NULL ||
			metadata->mount_src==NULL)
		{
			partition_metadata_free_elements(metadata);
			free(metadata);
			metadata=NULL;
		}
	}

	return metadata;
}

void partition_metadata_free(partition_t *partition)
{
	partition_metadata_t *metadata;
	partition_metadata_free_func_t free_func;

	metadata=partition_get_metadata(partition);
	free_func=metadata->free_func;

	partition_metadata_free_elements(metadata);

	free_func(metadata);
}

const char *partition_metadata_get_partition_identifier(partition_t *partition)
{
	partition_metadata_t *metadata;
	metadata=partition_get_metadata(partition);
	return metadata->identifier;
}

const char *partition_metadata_get_mountsrc(partition_t *partition)
{
	partition_metadata_t *metadata;
	metadata=partition_get_metadata(partition);
	return metadata->mount_src;
}

const char *partition_metadata_get_mountfs(partition_t *partition)
{
	partition_metadata_t *metadata;
	metadata=partition_get_metadata(partition);
	return metadata->mount_fs;
}

int partition_metadata_get_partition_number(partition_t *partition)
{
	partition_metadata_t *metadata;
	metadata=partition_get_metadata(partition);
	return metadata->partition_no;
}
//----------------------------------------------------------------------------------------------------------------

//-------------------------------------- private members ---------------------------------------------------------
static void partition_metadata_free_elements(partition_metadata_t *metadata)
{
	//sorry, this is needed to get rid of lint finding: "Custodial pointer ctx (line xxx) has not been freed or returned"
	partition_metadata_t *cp_metadata=metadata;
	if (cp_metadata->identifier!=NULL)
			free(cp_metadata->identifier);
	if (cp_metadata->mount_fs!=NULL)
			free(cp_metadata->mount_fs);
	if (cp_metadata->mount_src!=NULL)
			free(cp_metadata->mount_src);
}

static void partition_metadata_default_free(partition_metadata_t *metadata)
{
	free(metadata);
}
//----------------------------------------------------------------------------------------------------------------
